<?php
/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * @since      1.0.0
 */
class AR_Display {

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      AR_Display_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {
        if ( defined( 'AR_DISPLAY_VERSION' ) ) {
            $this->version = AR_DISPLAY_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'ar-display';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - AR_Display_Loader. Orchestrates the hooks of the plugin.
     * - AR_Display_i18n. Defines internationalization functionality.
     * - AR_Display_Admin. Defines all hooks for the admin area.
     * - AR_Display_Public. Defines all hooks for the public side of the site.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies() {

        /**
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */
        require_once AR_DISPLAY_PLUGIN_DIR . 'includes/class-ardisplay-loader.php';

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once AR_DISPLAY_PLUGIN_DIR . 'includes/class-ardisplay-i18n.php';

        /**
         * The class responsible for checking license status and restricting features
         */
        require_once AR_DISPLAY_PLUGIN_DIR . 'includes/class-ardisplay-api-key-helper.php';


        /**
         * The class responsible for 3D model handling
         */
        require_once AR_DISPLAY_PLUGIN_DIR . 'admin/class-ardisplay-3d-model.php';

        /**
         * The class responsible for defining all actions that occur in the admin area.
         */
        require_once AR_DISPLAY_PLUGIN_DIR . 'admin/class-ardisplay-admin.php';

        /**
         * The class responsible for defining all actions that occur in the public-facing
         * side of the site.
         */
        require_once AR_DISPLAY_PLUGIN_DIR . 'public/class-ardisplay-public.php';

    
        /**
         * The class responsible for blocks functionality
         */
        require_once AR_DISPLAY_PLUGIN_DIR . 'includes/class-ardisplay-blocks.php';

        $this->loader = new AR_Display_Loader();
        
        // Initialize the product integration
        
        // Initialize blocks
        new AR_Display_Blocks();
    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the AR_Display_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale() {
        $plugin_i18n = new AR_Display_i18n();
        $plugin_i18n->set_domain($this->get_plugin_name());

        $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks() {
        $plugin_admin = $this->get_plugin_admin();

        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');
        $this->loader->add_action('admin_notices', $this, 'display_api_key_warning');

        // Add menu items
        $this->loader->add_action('admin_menu', $plugin_admin, 'add_plugin_admin_menu');

        // Initialize the license helper
        AR_Display_api_key_Helper::init();
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks() {
        $plugin_public = new AR_Display_Public($this->get_plugin_name(), $this->get_version());

        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_styles');
        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_scripts');
        
    }

    /**
     * Display a warning notice if API key is not set.
     *
     * @since    1.0.0
     * @return   void
     */
    public function display_api_key_warning() {
        // Check if the API key is not set
        if ( ! AR_Display_api_key_Helper::has_api_key() && isset($_GET['page']) && $_GET['page'] !== 'ardisplay'  && $_GET['page'] !== 'ardisplay-license') {
            $api_key_page_url = admin_url( 'admin.php?page=ardisplay-license' );
            ?>
            <div class="notice notice-warning is-dismissible" style="margin-left: 0;">
                <p><?php 
                    printf( 
                        __( 'ARDisplay requires an API key to function. Please <a href="%s">configure your API key</a> to enable all features.', 'ardisplay' ), 
                        esc_url( $api_key_page_url ) 
                    ); 
                ?></p>
            </div>
            <?php
        }
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    AR_Display_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }

    /**
     * Retrieve the plugin admin instance.
     *
     * @since     1.0.0
     * @return    AR_Display_Admin    The plugin admin instance.
     */
    public function get_plugin_admin() {
        return new AR_Display_Admin($this->get_plugin_name(), $this->get_version());
    }
}
