<?php
/**
 * ardisplay Blocks
 *
 * @link       https://ardisplay.com
 * @since      1.0.0
 *
 * @package    ardisplay
 * @subpackage ardisplay/includes
 */

/**
 * Class responsible for registering and handling ardisplay blocks
 */
class AR_Display_Blocks {

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     */
    public function __construct() {
        // Enqueue scripts for block editor
        add_action('enqueue_block_editor_assets', array($this, 'enqueue_block_editor_assets'));
        
        // Enqueue frontend scripts
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_scripts'));

        $position = get_option('ardisplay_display_position', 'after_add_to_cart');
        
        switch ($position) {
            case 'before_add_to_cart':
                add_action('woocommerce_before_add_to_cart_button', array($this, 'add_viewer_block_to_product'));
                break;
            case 'before_image':
                add_action('woocommerce_product_thumbnails', array($this, 'add_viewer_block_to_product'), 20);
                break;
            case 'after_add_to_cart':
            default:
                add_action('woocommerce_after_add_to_cart_button', array($this, 'add_viewer_block_to_product'));
                break;
        }
                
    }


    /**
     * Enqueue block editor assets
     */
    /**
     * Enqueue block editor assets using WordPress standard functions.
     */
    public function enqueue_block_editor_assets() {
        $plugin_dir = dirname(dirname(__FILE__)); // Assumes this file is in includes/
        $plugin_url = plugins_url('', $plugin_dir . '/ardisplay.php'); // Adjust '/ardisplay.php' if your main plugin file has a different name

        $script_asset_path = $plugin_dir . '/build/viewer-block.asset.php'; // Path for generated asset file (if using @wordpress/scripts)
        $script_asset = file_exists($script_asset_path)
            ? require($script_asset_path)
            : array('dependencies' => array('wp-blocks', 'wp-element', 'wp-block-editor', 'wp-components', 'wp-i18n'), 'version' => filemtime($plugin_dir . '/blocks/viewer-block.js')); // Fallback dependencies and version


        // Enqueue the main AR Display Bridge (needed for the component)
        // We still need it loaded, give it a handle
        wp_enqueue_script(
            'ardisplay-bridge-lib',
            'https://cdn.statically.io/gh/ardisplaycorp/bridge/v2.0.0/build/ardisplay.es.min.js',
            array(), // No WP dependencies
            '1.9.9', // Version number
            true // Load in footer (doesn't strictly matter for modules like this, but good practice)
        );

        // Enqueue your block's script
        wp_enqueue_script(
            'ardisplay-viewer-block-editor', // Unique handle for the editor script
            $plugin_url . '/blocks/viewer-block.js', // Path to your block's JS file
            // Ensure the bridge lib is listed as a dependency so WP tries to load it first (though module loading handles actual timing)
            // Also add standard WP block dependencies
            array_merge($script_asset['dependencies'], ['ardisplay-bridge-lib']),
            $script_asset['version'],
            true // Load in footer
        );

        // Localize data for your block editor script
        wp_localize_script(
            'ardisplay-viewer-block-editor', // Use the *handle* of the script that needs the data
            'ardisplayModelsData', // The JavaScript object name
            array(
                // Ensure the API Key Helper class and method are available here
                'secretKey'  => class_exists('AR_Display_api_key_Helper') ? AR_Display_api_key_Helper::get_api_key() : '', // Add check
                'apiBaseUrl' => ARDISPLAY_API_ENDPOINT,
                 // Add nonce for security if making authenticated requests from editor
                // 'nonce'      => wp_create_nonce( 'wp_rest' )
            )
        );
    }

    /**
     * Enqueue frontend scripts
     */
    public function enqueue_frontend_scripts() {
        // Add model-viewer script tag directly with type="module"
        add_action('wp_head', function() {
            ?>
            <script type="module" src="https://cdn.statically.io/gh/ardisplaycorp/bridge/v2.0.0/build/ardisplay.es.min.js"></script>
            <?php
        });
    }



    /**
     * Adds the AR Display Viewer block to the single product page summary
     * if a 'model_id' custom field is set for the product.
     */
    public function add_viewer_block_to_product() {
        // // Ensure WooCommerce functions are available and we're on a product page
        if (!function_exists('is_product') || !is_product()) {
            return;
        }

        // // Get the global product object
        global $product;

        // // Double-check if $product is a valid WC_Product object
        if (!$product || !is_a($product, 'WC_Product')) {
             // Attempt to get product by ID as a fallback
             $product = wc_get_product(get_the_ID());
             if (!$product) {
                return; // Exit if we still can't get a product object
             }
        }

        // // Get the model ID from product meta data
        $model_id = $product->get_meta('_ardisplay_model_id', true); // Use WC_Product::get_meta for better compatibility

        // $ar_enabled = $product->get_meta('_ardisplay_enabled', true);

        $ar_enabled = get_post_meta($product->get_id(), '_ardisplay_enabled', true) ?? false;

        if (!empty($model_id) && $ar_enabled) {

            ?>
            <div class="ardisplay-viewer-wrapper" style="margin: 20px 0 20px 0;">
                <ardisplay-viewer
                    src="<?php echo $model_id; ?>"
                    class="ardisplay-viewer"
                    style="width: 100%; height: 100% !important;"
                    <?php if (get_option('ardisplay_display_full_width', 0)) : ?> fullWidth <?php endif; ?>
                    id="ardisplay-viewer-<?php echo esc_attr($model_id); ?>"
                ></ardisplay-viewer>
            </div>
            <?php
        }
    }
}
