<?php
/**
 * License Helper functionalities.
 *
 * @link       https://ardisplay.com
 * @since      1.0.0
 *
 * @package    ardisplay
 * @subpackage ardisplay/includes
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

/**
 * License Helper class.
 *
 * This class contains helper functions for license key management.
 *
 * @since      1.0.0
 * @package    ardisplay
 * @subpackage ardisplay/includes
 * @author     ardisplay Team <info@ardisplay.com>
 */
class AR_Display_api_key_Helper {

	/**
	 * The API base URL.
	 *
	 * @since    1.0.0
	 * @var      string    $api_base_url    The API base URL.
	 */
	private static $api_base_url = ARDISPLAY_API_ENDPOINT;

	/**
	 * Initialize the license helper.
	 * 
	 * @since    1.0.0
	 * @return   void
	 */
	public static function init() {
		// Setup is handled by the static methods
		// No initialization required
	}

	/**
	 * Get the license key from the database.
	 *
	 * @since    1.0.0
	 * @return   string    The license key or empty string if not set.
	 */
	public static function get_api_key() {
		return get_option( 'ardisplay_api_key', '' );
	}

	/**
	 * Check if a license key is set.
	 *
	 * @since    1.0.0
	 * @return   boolean    True if a license key is set, false otherwise.
	 */
	public static function has_api_key() {
		$api_key = self::get_api_key();
		return ! empty( $api_key );
	}

	/**
	 * Get the API URL for a given endpoint.
	 *
	 * @since    1.0.0
	 * @param    string    $endpoint    The endpoint to append to the API base URL.
	 * @return   string                The complete API URL.
	 */
	public static function get_api_url( $endpoint = '' ) {
		return rtrim( self::$api_base_url, '/' ) . '/' . ltrim( $endpoint, '/' );
	}

	/**
	 * Make an API request with the license key included in the parameters.
	 *
	 * @since    1.0.0
	 * @param    string    $endpoint    The API endpoint.
	 * @param    array     $params      The request parameters.
	 * @param    string    $method      The request method (GET, POST, PUT, or PATCH).
	 * @return   array|WP_Error         The API response or WP_Error on failure.
	 */
	public static function make_api_request( $endpoint, $params = array(), $method = 'GET' ) {
		$api_url = self::get_api_url( $endpoint );
		
		$args = array(
			'timeout' => 30,
			'headers' => array(
				'Accept' => 'application/json',
				'Content-Type' => 'application/json',
				'secret-key'=> self::get_api_key(),
			),
		);

		if ( $method === 'GET' ) {
			$api_url = add_query_arg( $params, $api_url );
			$response = wp_remote_get( $api_url, $args );
		} else if ( $method === 'POST' ) {
			$args['body'] = wp_json_encode( $params );
			$response = wp_remote_post( $api_url, $args );
		} else if ( $method === 'PUT' ) {
			$args['body'] = wp_json_encode( $params );
			$args['method'] = 'PUT';
			$response = wp_remote_request( $api_url, $args );
		} else if ( $method === 'PATCH' ) {
			$args['body'] = wp_json_encode( $params );
			$args['method'] = 'PATCH';
			$response = wp_remote_request( $api_url, $args );
		} else if ( $method === 'DELETE' ) {
			$args['body'] = wp_json_encode( $params );
			$args['method'] = 'DELETE';
			$response = wp_remote_request( $api_url, $args );
		} else {
			return new WP_Error( 'invalid_method', __( 'Invalid request method.', 'ardisplay' ) );
		}
		
		if ( is_wp_error( $response ) ) {

			return $response;
		}
		
		$response_code = wp_remote_retrieve_response_code( $response );
		
		if ( $response_code !== 200 && $response_code !== 201) {
			$message = wp_remote_retrieve_body( $response );
			return new WP_Error( 
				'api_error', 
				$message 
			);
		}
		
		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );
		
		if ( empty( $data ) ) {
			return new WP_Error( 'api_error', __( 'Invalid response format from the API.', 'ardisplay' ) );
		}
		
		return $data;
	}
}
