<?php if (!defined('ABSPATH')) exit; ?>
<script type="module">
    window.updateProgressUI = function(loaded, total) {
        const progressBar = document.getElementById("uploadProgressBar");
        const progressText = document.getElementById("uploadProgressText");
        const percentage = ((loaded / total) * 100).toFixed(2);
        progressBar.style.width = (percentage ?? 0) + "%";
        progressText.textContent = window.formatBytes(loaded) + " / " + window.formatBytes(total);
    };

    window.uploadFiles = async function(glbBlob, usdzBlob, posterBlob) {
        const result = {
            glb: null,
            usdz: null,
            poster: null
        };

        if (glbBlob) {
            try {
                result.glb = await window.uploadSingleFile(glbBlob, 'glb', 'model/gltf-binary' );
            } catch (error) {
                console.error('Error uploading GLB:', error);
            }
        }

        if (usdzBlob) {
            try {
                result.usdz = await window.uploadSingleFile(usdzBlob, 'usdz', 'model/usdz' );
            } catch (error) {
                console.error('Error uploading USDZ:', error);
            }
        }

        if (posterBlob) {
            try {
                result.poster = await window.uploadSingleFile(posterBlob, 'jpg', 'image/jpeg' );
            } catch (error) {
                console.error('Error uploading poster:', error);
            }
        }

        return result;
    };

    /**
     * Update document status from pending to completed
     * @param {string|null} documentId - Single document ID or null for batch update
     * @param {Array|null} documentIds - Array of document IDs for batch update
     * @returns {Promise} - Promise resolving to the API response
     */
    window.updateDocumentStatus = async function(documentIds = null , status = 'pending') {
        return new Promise((resolve, reject) => {
            try {
                const endpoint = `/documents/batch/status`;
                
                const xhr = new XMLHttpRequest();
                xhr.open("PUT", <?php echo json_encode(ASSETGER_API_ENDPOINT); ?> + endpoint);
                xhr.setRequestHeader("X-Secret-Key", <?php echo json_encode(ASSETGER_SECRET_KEY); ?>);
                xhr.setRequestHeader("Content-Type", "application/json");
                xhr.setRequestHeader("X-Requested-With", "XMLHttpRequest");

                xhr.onload = function() {
                    if (xhr.status >= 200 && xhr.status < 300) {
                        try {
                            const response = JSON.parse(xhr.response);
                            resolve(response);
                        } catch (e) {
                            reject("Invalid response format");
                        }
                    } else {
                        reject(xhr.statusText);
                    }
                };

                xhr.onerror = function() {
                    reject(xhr.statusText);
                };

                // Prepare request body
                const requestBody = {status};
                
                // Add document_ids for batch update
                if (documentIds && Array.isArray(documentIds)) {
                    requestBody.document_ids = documentIds;
                }

                xhr.send(JSON.stringify(requestBody));
            } catch (error) {
                reject(error);
            }
        });
    };


    window.generateUploadToken = async function() {
        const tokenResponse = await fetch(<?php echo json_encode(ARDISPLAY_API_ENDPOINT . "/upload-token"); ?>, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'Secret-Key': <?php echo json_encode(AR_Display_api_key_Helper::get_api_key()); ?>
            }
        });

        if (!tokenResponse.ok) {
            throw new Error(`Failed to get upload token: ${tokenResponse.statusText}`);
        }

        const tokenData = await tokenResponse.json();
        const uploadToken = tokenData.upload_token;
        
        if (!uploadToken) {
            throw new Error('Invalid upload token received');
        }
        
        return uploadToken;
    };

    window.uploadSingleFile = async function(file, extension, mimeType , token = null) {
        try {
            // Get an upload token
            const uploadToken = token ?? await window.generateUploadToken();
            
            // Now upload the file with the token
            return new Promise((resolve, reject) => {
                var xhr = new XMLHttpRequest();
                window.uploadXhr = xhr;

                xhr.open("POST", <?php echo json_encode(ASSETGER_API_ENDPOINT . "/documents/store"); ?>);
                //xhr.setRequestHeader("X-Secret-Key", <?php echo json_encode(ASSETGER_SECRET_KEY); ?>);
                xhr.setRequestHeader("X-Upload-Token", uploadToken);
                xhr.setRequestHeader("X-Requested-With", "XMLHttpRequest");

                var formData = new FormData();
                formData.append("file", file, `model.${extension}`);

                xhr.upload.onprogress = function(e) {
                    if (e.lengthComputable) {
                        window.updateProgressUI(e.loaded, e.total);
                    }
                };

                xhr.onload = function() {
                    if (xhr.status >= 200 && xhr.status < 300) {
                        try {
                            const response = JSON.parse(xhr.response);
                            resolve(response);
                        } catch (e) {
                            reject("Invalid response format");
                        }
                    } else {
                        reject(xhr.statusText);
                    }
                };

                xhr.onerror = function() {
                    reject(xhr.statusText);
                };

                xhr.send(formData);
            });
            
        } catch (error) {
            console.error("Upload error:", error);
            throw error;
        }
    };

    window.goToStep6AndUpload = async function() {
        if (window.isUploading) {
            return;
        }

        if (!window.finalGLBBlob || !window.finalUSDZBlob) {
            alert("Missing required files. Please try again.");
            return;
        }

        let hiddenViewer; // Declare hiddenViewer at the start of the function

        try {
            window.isUploading = true;
            
            // Create a hidden model-viewer for poster generation
            hiddenViewer = document.createElement('model-viewer');
            hiddenViewer.style.position = 'fixed';
            hiddenViewer.style.top = '0px';
            hiddenViewer.style.left = '0px';
            hiddenViewer.style.width = '1000px';
            hiddenViewer.style.height = '1000px';
            hiddenViewer.style.visibility = 'hidden';
            hiddenViewer.setAttribute('interaction-prompt', 'none');
            hiddenViewer.setAttribute('auto-rotate', 'false');
            hiddenViewer.setAttribute('camera-controls', 'false');
            hiddenViewer.src = window.generatedModelUrl;
            document.body.appendChild(hiddenViewer);

            // Wait for model to load and generate poster
            await new Promise(resolve => hiddenViewer.addEventListener('load', resolve, { once: true }));
            const posterBlob = await hiddenViewer.toBlob({
                idealAspect: true,
                mimeType: 'image/png'
            });

            // Clean up hidden viewer
            document.body.removeChild(hiddenViewer);

            window.showStep(6);
            
            // Reset progress bar
            const progressBar = document.getElementById("uploadProgressBar");
            const progressText = document.getElementById("uploadProgressText");
            progressBar.style.width = "0%";
            progressText.textContent = "0 B / 0 B";

            // Upload all files in sequence
            const result = await window.uploadFiles(window.finalGLBBlob, window.finalUSDZBlob, posterBlob);
            
            // Update document statuses to completed after successful upload
            const docIds = [];
            if (result.glb && result.glb.document && result.glb.document.id) {
                docIds.push(result.glb.document.id);
            }
            if (result.usdz && result.usdz.document && result.usdz.document.id) {
                docIds.push(result.usdz.document.id);
            }
            if (result.poster && result.poster.document && result.poster.document.id) {
                docIds.push(result.poster.document.id);
            }
            
            if (result.glb && result.glb.document && result.glb.document.url) { 
                const modelUrl = result.glb.document.url;
                const usdzUrl = result.usdz && result.usdz.document ? result.usdz.document.url : '';
                const posterFileUrl = result.poster && result.poster.document ? result.poster.document.url : '';
                
                // Add or update the model in table with all URLs
                const newModel = {
                    fileType: window.currentFileType,
                    sizes: window.uploadedModelSizes || [],
                    url: modelUrl,
                    usdzUrl: usdzUrl,
                    posterFileUrl: posterFileUrl,
                    iosId: result.usdz?.document?.id,
                    posterFileId: result.poster?.document?.id,
                    androidId: result.glb?.document?.id,
                    iosUrl: usdzUrl,
                    model: result.glb?.document?.id,
                    name: result.glb?.document?.name ?? ""
                };
                
                // Only call addModelToTable if we're not in edit mode
                const modal = document.getElementById('upload3dModal');
                if (!modal.dataset.editingRow) {
                    window.addModelToTable(newModel);
                } else {
                    window.updateExistingRow(newModel);
                }

                // Log the model data format after upload completes
                const modelData = {
                    addToCartUrl: document.getElementById('addToCartUrl').value,
                    description: document.getElementById('model_description').value,
                    id: document.querySelector('input[name="record_id"]')?.value || window.uniqid(),
                    title: document.getElementById('model_title').value,
                    url: document.getElementById('model_url').value,
                    variants: Array.from(document.querySelectorAll('#uploadedModelsTable tbody tr:not(.no-items)')).map((row, index) => {
                        const url = row.querySelector('.previewBtn').dataset.url;
                        const fileType = row.cells[1].textContent;
                        const sizes = JSON.parse(row.dataset.sizes || '[]');
                        
                        return {
                            name: `Variant ${index + 1}`,
                            model: (index + 1).toString(),
                            iosUrl: usdzUrl,
                            iosId: result.usdz?.document?.id,
                            posterFileUrl: posterFileUrl,
                            posterFileId: result.poster?.document?.id,
                            androidId: result.glb?.document?.id,
                            placement: fileType === 'Model' ? 'floor' : 'wall',
                            sizes: sizes,
                            url: url
                        };
                    })
                };
                console.log('Model Data:', modelData);
            }

            window.closeModal();

        } catch (error) {
            console.error('Error:', error);
            if (hiddenViewer && hiddenViewer.parentNode) {
                document.body.removeChild(hiddenViewer);
            }
            alert("Upload failed: " + error);
            window.closeModal();
        } finally {
            window.isUploading = false;
        }
    };
</script>
