/**
 * ardisplay QR Code Preview Script
 * 
 * Creates a live preview of QR code with various styling options
 */

(function($) {
    'use strict';
    
    // Initialize QR code preview on document ready
    $(document).ready(function() {
        // Check if QR code preview container exists
        if (!$('#ardisplay-qr-preview').length) {
            return;
        }
        
        // Initialize QR code object
        const qrCode = new QRCodeStyling({
            width: 250,
            height: 250,
            type: "svg",
            data: "https://ardisplay.com", // Default data
            dotsOptions: {
                color: "#000000",
                type: "square"
            },
            cornersSquareOptions: {
                color: "#000000",
                type: "square",
            },
            cornersDotOptions: {
                color: "#000000",
                type: "dot",
            },
            backgroundOptions: {
                color: "#ffffff",
            },
            imageOptions: {
                crossOrigin: "anonymous",
                margin: 3
            }
        });
        
        // Initial render
        qrCode.append(document.getElementById("ardisplay-qr-preview"));
        
        // Apply initial settings if they exist
        if (typeof ardisplayQrSettings !== 'undefined' && ardisplayQrSettings) {
            let settings = ardisplayQrSettings;
            
            // Set initial form values to match settings
            if (settings.baseUrl) {
                $('#ardisplay_qr_base_url').val(settings.baseUrl);
                qrCode.update({data: settings.baseUrl});
            }
            
            // Update dots style
            if (settings.dotStyle) {
                qrCode.update({
                    dotsOptions: {
                        type: settings.dotStyle
                    }
                });
            }
            
            // Update dots color
            if (settings.dotColor) {
                qrCode.update({
                    dotsOptions: {
                        color: settings.dotColor
                    }
                });
            }
            
            // Update corners style
            if (settings.cornerStyle) {
                qrCode.update({
                    cornersSquareOptions: {
                        type: settings.cornerStyle
                    }
                });
            }
            
            // Update corners color
            if (settings.cornerColor) {
                qrCode.update({
                    cornersSquareOptions: {
                        color: settings.cornerColor
                    }
                });
            }
            
            // Update corner dots style
            if (settings.cornerDotStyle) {
                qrCode.update({
                    cornersDotOptions: {
                        type: settings.cornerDotStyle
                    }
                });
            }
            
            // Update corner dots color
            if (settings.cornerDotColor) {
                qrCode.update({
                    cornersDotOptions: {
                        color: settings.cornerDotColor
                    }
                });
            }
            
            // Update background color
            if (settings.backgroundColor) {
                qrCode.update({
                    backgroundOptions: {
                        color: settings.backgroundColor
                    }
                });
            }
            
            // Update favicon/image
            if (settings.faviconUrl && settings.faviconUrl !== '') {
                qrCode.update({
                    image: settings.faviconUrl,
                    imageOptions: {
                        margin: parseInt(settings.faviconMargin) || 3
                    }
                });
            }
        }
        
        // Live preview updates
        
        // Update URL/data
        $('#ardisplay_qr_base_url').on('input change', function() {
            const url = $(this).val();
            qrCode.update({
                data: url
            });
        });
        
        // Update dot style
        $('#ardisplay_qr_dot_style').on('change', function() {
            qrCode.update({
                dotsOptions: {
                    type: $(this).val()
                }
            });
        });
        
        // Update dot color
        $('#ardisplay_qr_dot_color').on('input change', function() {
            qrCode.update({
                dotsOptions: {
                    color: $(this).val()
                }
            });
        });
        
        // Update corner style
        $('#ardisplay_qr_corner_style').on('change', function() {
            qrCode.update({
                cornersSquareOptions: {
                    type: $(this).val()
                }
            });
        });
        
        // Update corner color
        $('#ardisplay_qr_corner_color').on('input change', function() {
            qrCode.update({
                cornersSquareOptions: {
                    color: $(this).val()
                }
            });
        });
        
        // Update corner dot style
        $('#ardisplay_qr_corner_dot_style').on('change', function() {
            qrCode.update({
                cornersDotOptions: {
                    type: $(this).val()
                }
            });
        });
        
        // Update corner dot color
        $('#ardisplay_qr_corner_dot_color').on('input change', function() {
            qrCode.update({
                cornersDotOptions: {
                    color: $(this).val()
                }
            });
        });
        
        // Update background color
        $('#ardisplay_qr_background_color').on('input change', function() {
            qrCode.update({
                backgroundOptions: {
                    color: $(this).val()
                }
            });
        });
        
        // Update favicon/image
        let faviconUrlField = $('#ardisplay_qr_favicon_url');
        
        // Handle both direct URL changes and WordPress media upload
        faviconUrlField.on('input change', updateFavicon);
        
        function updateFavicon() {
            const faviconUrl = faviconUrlField.val();
            const faviconMargin = parseInt($('#ardisplay_qr_favicon_margin').val()) || 3;
            
            if (faviconUrl && faviconUrl !== '') {
                qrCode.update({
                    image: faviconUrl,
                    imageOptions: {
                        margin: faviconMargin
                    }
                });
            } else {
                // Remove image if URL is empty
                qrCode.update({
                    image: undefined
                });
            }
        }
        
        // Update favicon margin
        $('#ardisplay_qr_favicon_margin').on('input change', function() {
            const faviconUrl = faviconUrlField.val();
            const faviconMargin = parseInt($(this).val()) || 3;
            
            if (faviconUrl && faviconUrl !== '') {
                qrCode.update({
                    imageOptions: {
                        margin: faviconMargin
                    }
                });
            }
        });
        
        // Media uploader for favicon
        $('#ardisplay_qr_favicon_upload_button').on('click', function(e) {
            e.preventDefault();
            
            const mediaUploader = wp.media({
                title: 'Select or Upload QR Code Center Image',
                button: {
                    text: 'Use this image'
                },
                multiple: false
            });
            
            mediaUploader.on('select', function() {
                const attachment = mediaUploader.state().get('selection').first().toJSON();
                $('#ardisplay_qr_favicon_id').val(attachment.id);
                $('#ardisplay_qr_favicon_url').val(attachment.url).trigger('change');
            });
            
            mediaUploader.open();
        });
        
        // Download QR Code button
        $('#ardisplay_qr_download_button').on('click', function(e) {
            e.preventDefault();
            qrCode.download({
                extension: 'png'
            });
        });
        
        // QR Favicon Upload Handler
        $('#ardisplay_qr_favicon_upload').on('change', async function(e) {
            const file = e.target.files[0];
            if (!file) return;

            try {
                // Use the uploadSingleFile function from upload-handlers.js
                const response = await window.uploadSingleFile(file, file.name.split('.').pop(), file.type);
                
                if (response && response.document.id && response.document.url) {
                    // Set the URL input
                    $('#ardisplay_qr_favicon_url').val(response.document.url);
                    
                    // Set the ID input
                    $('#ardisplay_qr_favicon_id').val(response.document.id);


                    qrCode.update({
                        image: response.document.url,
                    });
                    
                } else {
                    console.error('Upload failed: Invalid response');
                    alert('Failed to upload favicon. Please try again.');
                }
            } catch (error) {
                console.error('Favicon upload error:', error);
                alert('Failed to upload favicon. Please try again.');
            }
        });
    });
})(jQuery);
