<?php
/**
 * 3D Models table display page
 *
 * @link       https://ardisplay.com
 * @since      1.0.0
 *
 * @package    Ardisplay
 * @subpackage Ardisplay/admin/partials
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

// Include the API helper if not already included
if (!class_exists('AR_Display_api_key_Helper') && defined('AR_DISPLAY_PLUGIN_DIR')) {
    require_once AR_DISPLAY_PLUGIN_DIR . 'includes/class-ardisplay-api-key-helper.php';
}

// Fetch models data from API - default parameters
$current_page = isset($_GET['page_num']) ? intval($_GET['page_num']) : 1;
$per_page = isset($_GET['per_page']) ? intval($_GET['per_page']) : 10;
$search = isset($_GET['search']) ? sanitize_text_field($_GET['search']) : '';

// Prepare API request parameters
$params = array(
    'page' => $current_page,
    'perPage' => $per_page
);

// Add search parameter if not empty
if (!empty($search)) {
    $params['search'] = $search;
}

// Fetch data from the API
$models_data = AR_Display_api_key_Helper::make_api_request('3d-model', $params, 'GET');

// Enqueue styles and scripts for this page
wp_enqueue_style('ardisplay-models-table', AR_DISPLAY_PLUGIN_URL . '/assets/css/ardisplay-models-table.css', array(), '1.0.0');
wp_enqueue_script('ardisplay-models-table', AR_DISPLAY_PLUGIN_URL . '/assets/js/ardisplay-models-table.js', array('jquery'), '1.0.0', true);

// Localize script with necessary data
wp_localize_script('ardisplay-models-table', 'ardisplayModelsData', array(
    'ajaxurl' => admin_url('admin-ajax.php'),
    'nonce' => wp_create_nonce('ardisplay_3d_model_nonce'),
    'currentPage' => $current_page,
    'perPage' => $per_page,
    'search' => $search
));

?>

<div class="wrap">
    <div style="display: flex; justify-content: space-between; align-items: center;">
        <h1 class="wp-heading-inline"><?php echo esc_html__('3D Models', 'ardisplay'); ?></h1>
        <a href="<?php echo esc_url(admin_url('admin.php?page=ardisplay-configure-ar')); ?>" class="page-title-action"><?php echo esc_html__('Add New Model', 'ardisplay'); ?></a>
    </div>
    
    <hr class="wp-header-end">
    
    <div class="notice notice-info">
        <p><?php echo esc_html__('Manage your 3D models for AR product visualization. These models can be assigned to products to enable AR previews.', 'ardisplay'); ?></p>
    </div>
    
    <form method="get" action="" id="ardisplay-models-filter">
        <input type="hidden" name="page" value="ardisplay-models">
        
        <!-- Search Box -->
        <p class="search-box">
            <label class="screen-reader-text" for="search"><?php echo esc_html__('Search Models:', 'ardisplay'); ?></label>
            <input type="search" id="search" name="search" value="<?php echo esc_attr($search); ?>" placeholder="<?php echo esc_attr__('Search models...', 'ardisplay'); ?>">
            <input type="submit" id="search-submit" class="button" value="<?php echo esc_attr__('Search Models', 'ardisplay'); ?>">
        </p>
        
        <!-- Models Table -->
        <div class="tablenav top">
            <div class="alignleft actions bulkactions">
                <label for="per_page" class="screen-reader-text"><?php echo esc_html__('Number of items per page:', 'ardisplay'); ?></label>
                <select name="per_page" id="per_page" class="postform">
                    <option value="10" <?php selected($per_page, 10); ?>>10</option>
                    <option value="25" <?php selected($per_page, 25); ?>>25</option>
                    <option value="50" <?php selected($per_page, 50); ?>>50</option>
                    <option value="100" <?php selected($per_page, 100); ?>>100</option>
                </select>
                <input type="submit" id="doaction" class="button action" value="<?php echo esc_attr__('Apply', 'ardisplay'); ?>">
            </div>
            
            <?php if (isset($models_data['totalPages']) && $models_data['totalPages'] > 0) : ?>
            <div class="tablenav-pages">
                <span class="displaying-num">
                    <?php 
                    if (isset($models_data['totalItems'])) {
                        printf(
                            _n('%s item', '%s items', $models_data['totalItems'], 'ardisplay'),
                            number_format_i18n($models_data['totalItems'])
                        );
                    }
                    ?>
                </span>
                
                <span class="pagination-links">
                    <?php
                    $total_pages = $models_data['totalPages'];
                    
                    // First page link
                    if ($current_page > 1) {
                        printf(
                            '<a class="first-page button" href="%s"><span class="screen-reader-text">%s</span><span aria-hidden="true">«</span></a>',
                            esc_url(add_query_arg(array('page' => 'ardisplay-models', 'page_num' => 1, 'per_page' => $per_page, 'search' => $search), admin_url('admin.php'))),
                            esc_html__('First page', 'ardisplay')
                        );
                        
                        printf(
                            '<a class="prev-page button" href="%s"><span class="screen-reader-text">%s</span><span aria-hidden="true">‹</span></a>',
                            esc_url(add_query_arg(array('page' => 'ardisplay-models', 'page_num' => max(1, $current_page - 1), 'per_page' => $per_page, 'search' => $search), admin_url('admin.php'))),
                            esc_html__('Previous page', 'ardisplay')
                        );
                    } else {
                        echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">«</span>';
                        echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">‹</span>';
                    }
                    
                    printf(
                        '<span class="paging-input"><input class="current-page" id="current-page-selector" type="text" name="page_num" value="%s" size="1" aria-describedby="table-paging"> <span class="tablenav-paging-text"> %s <span class="total-pages">%s</span></span></span>',
                        esc_attr($current_page),
                        esc_html__('of', 'ardisplay'),
                        esc_html($total_pages)
                    );
                    
                    // Next page link
                    if ($current_page < $total_pages) {
                        printf(
                            '<a class="next-page button" href="%s"><span class="screen-reader-text">%s</span><span aria-hidden="true">›</span></a>',
                            esc_url(add_query_arg(array('page' => 'ardisplay-models', 'page_num' => min($total_pages, $current_page + 1), 'per_page' => $per_page, 'search' => $search), admin_url('admin.php'))),
                            esc_html__('Next page', 'ardisplay')
                        );
                        
                        printf(
                            '<a class="last-page button" href="%s"><span class="screen-reader-text">%s</span><span aria-hidden="true">»</span></a>',
                            esc_url(add_query_arg(array('page' => 'ardisplay-models', 'page_num' => $total_pages, 'per_page' => $per_page, 'search' => $search), admin_url('admin.php'))),
                            esc_html__('Last page', 'ardisplay')
                        );
                    } else {
                        echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">›</span>';
                        echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">»</span>';
                    }
                    ?>
                </span>
            </div>
            <?php endif; ?>
            <br class="clear">
        </div>
        
        <table class="wp-list-table widefat fixed striped posts">
            <thead>
                <tr>
                    <th scope="col" class="manage-column column-thumbnail"></th>
                    <th scope="col" class="manage-column column-title column-primary"><?php echo esc_html__('Title', 'ardisplay'); ?></th>
                    <th scope="col" class="manage-column column-description"><?php echo esc_html__('Description', 'ardisplay'); ?></th>
                    <th scope="col" class="manage-column column-variants"><?php echo esc_html__('Variants', 'ardisplay'); ?></th>
                    <th scope="col" class="manage-column column-created"><?php echo esc_html__('Created', 'ardisplay'); ?></th>
                    <th scope="col" class="manage-column column-status"><?php echo esc_html__('Status', 'ardisplay'); ?></th>
                    <th scope="col" class="manage-column column-actions"><?php echo esc_html__('Actions', 'ardisplay'); ?></th>
                </tr>
            </thead>
            <tbody id="the-list">
                <?php
                if (isset($models_data['data']) && !empty($models_data['data'])) {
                    foreach ($models_data['data'] as $model) {
                        // Get the first variant's poster image for thumbnail
                        $thumbnail_url = '';
                        $variant_count = 0;
                        
                        if (isset($model['variants']) && !empty($model['variants'])) {
                            $variant_count = count($model['variants']);
                            
                            if (isset($model['variants'][0]['posterFileUrl']) && !empty($model['variants'][0]['posterFileUrl'])) {
                                $thumbnail_url = $model['variants'][0]['posterFileUrl'];
                            }
                        }
                        
                        // Format the creation date
                        $created_date = '';
                        if (isset($model['createdAt']) && !empty($model['createdAt'])) {
                            $date = new DateTime($model['createdAt']);
                            $created_date = $date->format(get_option('date_format'));
                        }
                        
                        // Get the model status
                        $status = isset($model['enabled']) && $model['enabled'] ? 'enabled' : 'disabled';
                        $status_label = isset($model['enabled']) && $model['enabled'] ? __('Enabled', 'ardisplay') : __('Disabled', 'ardisplay');
                        ?>
                        <tr data-model-id="<?php echo esc_attr($model['id']); ?>">
                            <td class="column-thumbnail">
                                <?php if (!empty($thumbnail_url)) : ?>
                                    <img src="<?php echo esc_url($thumbnail_url); ?>" alt="<?php echo esc_attr($model['title']); ?>" width="50" height="50">
                                <?php else : ?>
                                    <div class="no-thumbnail"><?php echo esc_html__('No Image', 'ardisplay'); ?></div>
                                <?php endif; ?>
                            </td>
                            <td class="column-title column-primary">
                                <strong><a href="<?php echo esc_url(admin_url('admin.php?page=ardisplay-configure-ar&id=' . $model['id'])); ?>"><?php echo esc_html($model['title']); ?></a></strong>
                                <div class="row-actions">
                                    <span class="edit">
                                        <a href="<?php echo esc_url(admin_url('admin.php?page=ardisplay-configure-ar&id=' . $model['id'])); ?>"><?php echo esc_html__('Edit', 'ardisplay'); ?></a> | 
                                    </span>
                                    <span class="trash">
                                        <a href="#" class="ardisplay-delete-model" data-model-id="<?php echo esc_attr($model['id']); ?>"><?php echo esc_html__('Delete', 'ardisplay'); ?></a>
                                    </span>
                                </div>
                                <button type="button" class="toggle-row"><span class="screen-reader-text"><?php echo esc_html__('Show more details', 'ardisplay'); ?></span></button>
                            </td>
                            <td class="column-description"><?php echo wp_trim_words(esc_html($model['description']), 10, '...'); ?></td>
                            <td class="column-variants"><?php echo esc_html($variant_count); ?></td>
                            <td class="column-created"><?php echo esc_html($created_date); ?></td>
                            <td class="column-status">
                                <span class="ardisplay-status ardisplay-status-<?php echo esc_attr($status); ?>"><?php echo esc_html($status_label); ?></span>
                            </td>
                            <td class="column-actions">
                                <a href="<?php echo esc_url(admin_url('admin.php?page=ardisplay-configure-ar&action=edit&id=' . $model['id'])); ?>" class="button-link"><?php echo esc_html__('Edit', 'ardisplay'); ?></a>
                                |
                                <button class="button-link ardisplay-delete-model" data-model-id="<?php echo esc_attr($model['id']); ?>"><?php echo esc_html__('Delete', 'ardisplay'); ?></button>
                            </td>
                        </tr>
                        <?php
                    }
                } else {
                    ?>
                    <tr>
                        <td colspan="7" class="colspanchange">
                            <p><?php echo esc_html__('No 3D models found.', 'ardisplay'); ?></p>
                            <p><a href="<?php echo esc_url(admin_url('admin.php?page=ardisplay-configure-ar')); ?>" class="button button-primary"><?php echo esc_html__('Create your first 3D model', 'ardisplay'); ?></a></p>
                        </td>
                    </tr>
                    <?php
                }
                ?>
            </tbody>
            <tfoot>
                <tr>
                    <th scope="col" class="manage-column column-thumbnail"><?php echo esc_html__('Thumbnail', 'ardisplay'); ?></th>
                    <th scope="col" class="manage-column column-title column-primary"><?php echo esc_html__('Title', 'ardisplay'); ?></th>
                    <th scope="col" class="manage-column column-description"><?php echo esc_html__('Description', 'ardisplay'); ?></th>
                    <th scope="col" class="manage-column column-variants"><?php echo esc_html__('Variants', 'ardisplay'); ?></th>
                    <th scope="col" class="manage-column column-created"><?php echo esc_html__('Created', 'ardisplay'); ?></th>
                    <th scope="col" class="manage-column column-status"><?php echo esc_html__('Status', 'ardisplay'); ?></th>
                    <th scope="col" class="manage-column column-actions"><?php echo esc_html__('Actions', 'ardisplay'); ?></th>
                </tr>
            </tfoot>
        </table>
        
        <?php if (isset($models_data['totalPages']) && $models_data['totalPages'] > 0) : ?>
        <div class="tablenav bottom">
            <div class="tablenav-pages">
                <span class="displaying-num">
                    <?php 
                    if (isset($models_data['totalItems'])) {
                        printf(
                            _n('%s item', '%s items', $models_data['totalItems'], 'ardisplay'),
                            number_format_i18n($models_data['totalItems'])
                        );
                    }
                    ?>
                </span>
                
                <span class="pagination-links">
                    <?php
                    // First page link
                    if ($current_page > 1) {
                        printf(
                            '<a class="first-page button" href="%s"><span class="screen-reader-text">%s</span><span aria-hidden="true">«</span></a>',
                            esc_url(add_query_arg(array('page' => 'ardisplay-models', 'page_num' => 1, 'per_page' => $per_page, 'search' => $search), admin_url('admin.php'))),
                            esc_html__('First page', 'ardisplay')
                        );
                        
                        printf(
                            '<a class="prev-page button" href="%s"><span class="screen-reader-text">%s</span><span aria-hidden="true">‹</span></a>',
                            esc_url(add_query_arg(array('page' => 'ardisplay-models', 'page_num' => max(1, $current_page - 1), 'per_page' => $per_page, 'search' => $search), admin_url('admin.php'))),
                            esc_html__('Previous page', 'ardisplay')
                        );
                    } else {
                        echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">«</span>';
                        echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">‹</span>';
                    }
                    
                    printf(
                        '<span class="paging-input"><input class="current-page" id="current-page-selector-bottom" type="text" name="page_num" value="%s" size="1" aria-describedby="table-paging"> <span class="tablenav-paging-text"> %s <span class="total-pages">%s</span></span></span>',
                        esc_attr($current_page),
                        esc_html__('of', 'ardisplay'),
                        esc_html($total_pages)
                    );
                    
                    // Next page link
                    if ($current_page < $total_pages) {
                        printf(
                            '<a class="next-page button" href="%s"><span class="screen-reader-text">%s</span><span aria-hidden="true">›</span></a>',
                            esc_url(add_query_arg(array('page' => 'ardisplay-models', 'page_num' => min($total_pages, $current_page + 1), 'per_page' => $per_page, 'search' => $search), admin_url('admin.php'))),
                            esc_html__('Next page', 'ardisplay')
                        );
                        
                        printf(
                            '<a class="last-page button" href="%s"><span class="screen-reader-text">%s</span><span aria-hidden="true">»</span></a>',
                            esc_url(add_query_arg(array('page' => 'ardisplay-models', 'page_num' => $total_pages, 'per_page' => $per_page, 'search' => $search), admin_url('admin.php'))),
                            esc_html__('Last page', 'ardisplay')
                        );
                    } else {
                        echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">›</span>';
                        echo '<span class="tablenav-pages-navspan button disabled" aria-hidden="true">»</span>';
                    }
                    ?>
                </span>
            </div>
            <br class="clear">
        </div>
        <?php endif; ?>
    </form>
</div>

<!-- Delete Model Confirmation Modal -->
<div id="ardisplay-delete-model-modal" style="display: none;">
    <div class="ardisplay-modal-content">
        <h2><?php echo esc_html__('Confirm Deletion', 'ardisplay'); ?></h2>
        <p><?php echo esc_html__('Are you sure you want to delete this 3D model? This action cannot be undone.', 'ardisplay'); ?></p>
        <div class="ardisplay-modal-actions">
            <button class="button ardisplay-cancel-delete"><?php echo esc_html__('Cancel', 'ardisplay'); ?></button>
            <button class="button button-primary ardisplay-confirm-delete"><?php echo esc_html__('Delete', 'ardisplay'); ?></button>
        </div>
    </div>
</div>
