<?php
/**
 * 3D Model functionality for the ARDisplay plugin
 *
 * @link       https://ardisplay.com
 * @since      1.0.0
 *
 * @package    ARDisplay
 * @subpackage ARDisplay/admin
 */

/**
 * The admin-specific 3D model functionality of the plugin.
 *
 * Handles the admin-facing aspects of 3D model management, including
 * storing and updating models through AJAX requests.
 *
 * @package    ARDisplay
 * @subpackage ARDisplay/admin
 */
class AR_Display_Settings {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       Optional. The name of this plugin.
     * @param    string    $version           Optional. The version of this plugin.
     */
    public function __construct($plugin_name = 'ardisplay', $version = '1.0.0') {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        
        // Register AJAX actions for settings updates
        add_action('wp_ajax_ardisplay_update_settings', array($this, 'update_settings_ajax'));
        add_action('wp_ajax_nopriv_ardisplay_update_settings', array($this, 'update_settings_ajax'));

       }

    public function get_settings() {
        return AR_Display_api_key_Helper::make_api_request('customize', [], 'GET');
    }



     /**
     * Updates plugin settings
     *
     * @param array $settings The settings data to update
     * @return boolean|WP_Error True on success, WP_Error on failure
     */
    public function update_settings($settings) {
        // Verify that settings is valid
        if (!is_array($settings)) {
            return new WP_Error('invalid_settings', __('Invalid settings data provided.', 'ardisplay'));
        }


        // Prepare endpoint URL
        $valid_settings = array();

        // Handle button settings
        if (isset($settings['form']) && $settings['form'] === 'button') {
            $valid_settings = array(
                'form' => 'button',
                'btnText' => isset($settings['btnText']) ? sanitize_text_field($settings['btnText']) : 'View in AR',
                'btnBgColor' => isset($settings['btnBgColor']) ? sanitize_hex_color($settings['btnBgColor']) : '#000000',
                'btnTextColor' => isset($settings['btnTextColor']) ? sanitize_hex_color($settings['btnTextColor']) : '#ffffff',
                'cornerRadius' => isset($settings['cornerRadius']) ? (string)absint($settings['cornerRadius']) : '4',
                'btnSize' => isset($settings['btnSize']) ? (string)absint($settings['btnSize']) : '18',
                'btnIcon' => isset($settings['btnIcon']) ? sanitize_text_field($settings['btnIcon']) : 'Eye',
            );
        }

        // Handle QR code settings - matches QRAttrSchema validation
        if (isset($settings['form']) && $settings['form'] === 'qr') {
            $valid_settings = array(
                'form' => 'qr',
                'faviconId' => isset($settings['faviconId']) ? sanitize_text_field($settings['faviconId']) : '',
                'faviconUrl' => isset($settings['faviconUrl']) ? esc_url_raw($settings['faviconUrl']) : '',
                'faviconMargin' => isset($settings['faviconMargin']) ? sanitize_text_field($settings['faviconMargin']) : '3',
                'QRsize' => isset($settings['size']) ? (string)absint($settings['size']) : '150',
                'dotStyle' => isset($settings['dotStyle']) ? sanitize_text_field($settings['dotStyle']) : 'square',
                'dotColor' => isset($settings['dotColor']) ? sanitize_hex_color($settings['dotColor']) : '#000000',
                'cornerStyle' => isset($settings['cornerStyle']) ? sanitize_text_field($settings['cornerStyle']) : 'square',
                'cornerColor' => isset($settings['cornerColor']) ? sanitize_hex_color($settings['cornerColor']) : '#000000',
                'cornerDotStyle' => isset($settings['cornerDotStyle']) ? sanitize_text_field($settings['cornerDotStyle']) : 'dot',
                'cornerDotColor' => isset($settings['cornerDotColor']) ? sanitize_hex_color($settings['cornerDotColor']) : '#000000',
                'backgroundColor' => isset($settings['backgroundColor']) ? sanitize_hex_color($settings['backgroundColor']) : '#ffffff',
                'imgBackground' => isset($settings['imgBackground']) ? (bool)($settings['imgBackground']) : true,
                'website' => isset($settings['baseUrl']) ? esc_url_raw($settings['baseUrl']) : get_site_url(),
            );
        }

        // Handle display mode settings
        if (isset($settings['form']) && $settings['form'] === 'mode') {
            $valid_settings = array(
                'form' => 'mode',
                'glbee' => isset($settings['glbee']) ? sanitize_text_field($settings['glbee']) : 'none',
                'qrCode' => isset($settings['qrCode']) ? sanitize_text_field($settings['qrCode']) : 'popup',
            );
        }

        //Handle display product settings
        if (isset($settings['form']) && $settings['form'] === 'product') {
            $valid_settings = array(
                'position' => isset($settings['position']) ? sanitize_text_field($settings['position']) : 'before_add_to_cart',
                'fullWidth' => isset($settings['fullWidth']) ? (bool)($settings['fullWidth']) : false,
            );

            // Save local WordPress options
            update_option('ardisplay_display_position', $valid_settings['position']);
            update_option('ardisplay_display_full_width', $valid_settings['fullWidth'] ? 1 : 0);


            // Return success response
            return array('success' => true, 'message' => __('Display settings saved successfully.', 'ardisplay'));
        }
          
        return AR_Display_api_key_Helper::make_api_request('customize', $valid_settings, 'POST');
    }

    /**
     * AJAX handler for updating settings
     * 
     * Handles AJAX requests to update plugin settings
     * 
     * @since 1.0.0
     * @return void Outputs JSON response and exits
     */
    public function update_settings_ajax() {
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have permission to update settings.', 'ardisplay')), 403);
        }
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'ardisplay_settings_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed. Please refresh the page and try again.', 'ardisplay')), 401);
        }
        
        // Get settings data
        $settings = isset($_POST['settings']) ? $_POST['settings'] : array();
        
        // If settings is a JSON string, decode it
        if (is_string($settings) && !empty($settings)) {
            $decoded_settings = json_decode($settings, true);
            if (json_last_error() === JSON_ERROR_NONE) {
                $settings = $decoded_settings;
            }
        }
        
        $result = $this->update_settings($settings);

        if (is_wp_error($result)) {
            wp_send_json_error(array('message' => $result->get_error_message()), 400);
        }
        
        // Return success response
        wp_send_json_success(array('message' => __('Settings saved successfully.', 'ardisplay')));
    }

    /**
     * Enqueue scripts for AJAX functionality
     * 
     * @since    1.0.0
     * @param    string    $hook    The current admin page
     */
    public function enqueue_scripts($hook) {
        // Enqueue Axios if not already loaded
        wp_enqueue_script('axios', 'https://cdn.jsdelivr.net/npm/axios/dist/axios.min.js', array(), '0.21.1', true);

        // Enqueue custom script for display settings
        wp_enqueue_script(
            'ardisplay-display-settings', 
            plugin_dir_url(__FILE__) . 'js/ardisplay-display-settings.js', 
            array('axios'), 
            $this->version, 
            true
        );

        // Localize script with nonce and ajax url
        wp_localize_script('ardisplay-display-settings', 'ardisplayDisplaySettings', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('ardisplay_display_settings_nonce')
        ));
    }
}
