    <?php
/**
 * WooCommerce Product List Table Class for ARDISPLAY
 *
 * @since      1.0.0
 */

// If this file is called directly, abort.
if (!defined('WPINC')) {
    die;
}

if (!class_exists('WP_List_Table')) {
    require_once(ABSPATH . 'wp-admin/includes/class-wp-list-table.php');
}

class AR_Display_Product_List_Table extends WP_List_Table {

    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct(array(
            'singular' => 'product',
            'plural'   => 'products',
            'ajax'     => false
        ));

    }

    /**
     * Get columns
     */ 
    public function get_columns() {
        $columns = array(
            'cb'           => '<input type="checkbox" />',
            'image'        => __('Image', 'ardisplay'),
            'title'        => __('Product Name', 'ardisplay'),
            'ar_enabled'   => __('AR Enabled', 'ardisplay'),
            'model_id'     => __('Model ID', 'ardisplay'),
            'configure'    => __('Configure AR', 'ardisplay'),
        );
        return $columns;
    }

    /**
     * Get sortable columns
     */
    public function get_sortable_columns() {
        $sortable_columns = array(
            'title'  => array('title', true),
        );
        return $sortable_columns;
    }

    /**
     * Default column rendering
     */
    public function column_default($item, $column_name) {
        switch ($column_name) {
            case 'ar_enabled':
                $product_id = $item->get_id();
                $nonce = wp_create_nonce('ardisplay_toggle_ar_' . $product_id);
                $is_ar_enabled = get_post_meta($product_id, '_ardisplay_enabled', true);
                $checked = $is_ar_enabled ? 'checked' : '';

                return '
                    <label class="ardisplay-toggle-switch">
                        <input type="checkbox" class="ar-toggle" 
                               data-product-id="' . esc_attr($product_id) . '" 
                               data-nonce="' . esc_attr($nonce) . '" 
                               ' . $checked . '>
                        <span class="ardisplay-toggle-slider"></span>
                    </label>
                ';
            case 'model_id':
                return $this->render_ardisplay_column('_ardisplay_model', $item->get_id()); 
            case 'configure':
                $model_id = get_post_meta($item->get_id(), '_ardisplay_model_id', true);
                $url = wp_nonce_url(
                    admin_url(
                        'admin.php?page=ardisplay-configure-ar' .
                        ( !empty($model_id) ? '&action=edit&id=' . $model_id : '' ) .
                        '&product_id=' . $item->get_id()
                    ),
                    'configure_ar_' . $item->get_id()
                );
                return '<a href="' . esc_url($url) . '" class="button button-small ardisplay-configure-button" style="line-height: 1.2;" title="' . esc_attr__('Configure ARt Preview', 'ardisplay') . '">
                    <span class="dashicons dashicons-admin-generic ardisplay-config-icon"></span>
                </a>';
            default:
                return '';
        }
    }

    /**
     * Column cb
     */
    public function column_cb($item) {
        return sprintf(
            '<input type="checkbox" name="product[]" value="%s" />',
            $item->get_id()
        );
    }

    /**
     * Column image
     */
    public function column_image($item) {
        $image_id = $item->get_image_id();
        if ($image_id) {
            $image = wp_get_attachment_image($image_id, array(50, 50));
        } else {
            $image = wc_placeholder_img(array(50, 50));
        }
        return $image;
    }

    /**
     * Column title
     */
    public function column_title($item) {
        $actions = array('enable_ar'  => sprintf('<a href="%s">%s</a>', 
                wp_nonce_url(admin_url('admin.php?page=ardisplay&action=enable_ar&product_id=' . $item->get_id()), 'enable_ar_' . $item->get_id()), 
                get_post_meta($item->get_id(), '_ardisplay_enabled', true) === 'yes' ? __('Disable AR Preview', 'ardisplay') : __('Enable AR Preview', 'ardisplay')
            ),
            'configure_ar' => sprintf('<a href="%s">%s</a>', 
                wp_nonce_url(admin_url('admin.php?page=ardisplay-configure-ar&product_id=' . $item->get_id()), 'configure_ar_' . $item->get_id()),
                __('Configure ARt Preview', 'ardisplay')
            ),
        );

        return sprintf(
            '<strong><a href="%s">%s</a></strong> %s',
            get_edit_post_link($item->get_id()),
            $item->get_name(),
            $this->row_actions($actions)
        );
    }

    /**
     * Process bulk actions
     */
    public function process_bulk_action() {
        if ('enable_ar' === $this->current_action()) {
            // Process single product AR enable
            if (isset($_GET['product_id']) && isset($_GET['_wpnonce'])) {
                $product_id = intval($_GET['product_id']);
                
                if (wp_verify_nonce($_GET['_wpnonce'], 'enable_ar_' . $product_id)) {
                    // Toggle the AR status
                    $current_status = get_post_meta($product_id, '_ardisplay_enabled', true);
                    $new_status = ($current_status === 'yes') ? 'no' : 'yes';
                    update_post_meta($product_id, '_ardisplay_enabled', $new_status);
                    
                    // Add admin notice
                    add_action('admin_notices', function() use ($new_status) {
                        $message = $new_status === 'yes' ? __('AR preview enabled for the product.', 'ardisplay') : __('AR preview disabled for the product.', 'ardisplay');
                        echo '<div class="notice notice-success is-dismissible"><p>' . $message . '</p></div>';
                    });
                }
            }
            
            // Process bulk AR enable
            if (isset($_POST['product']) && is_array($_POST['product'])) {
                // Check nonce
                check_admin_referer('bulk-' . $this->_args['plural']);
                
                $product_ids = array_map('intval', $_POST['product']);
                foreach ($product_ids as $product_id) {
                    update_post_meta($product_id, '_ardisplay_enabled', 'yes');
                }
                
                // Add admin notice
                add_action('admin_notices', function() {
                    echo '<div class="notice notice-success is-dismissible"><p>' . __('AR previews enabled for the selected products.', 'ardisplay') . '</p></div>';
                });
            }
        }
        
        if ('disable_ar' === $this->current_action() && isset($_POST['product']) && is_array($_POST['product'])) {
            // Check nonce
            check_admin_referer('bulk-' . $this->_args['plural']);
            
            $product_ids = array_map('intval', $_POST['product']);
            foreach ($product_ids as $product_id) {
                update_post_meta($product_id, '_ardisplay_enabled', 'no');
            }
            
            // Add admin notice
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p>' . __('AR previews disabled for the selected products.', 'ardisplay') . '</p></div>';
            });
        }
        
        if ('clear_model_id' === $this->current_action() && isset($_POST['product']) && is_array($_POST['product'])) {
            // Check nonce
            check_admin_referer('bulk-' . $this->_args['plural']);
            
            $product_ids = array_map('intval', $_POST['product']);
            foreach ($product_ids as $product_id) {
                delete_post_meta($product_id, '_ardisplay_model_id');
            }
            
            // Add admin notice
            add_action('admin_notices', function() {
                echo '<div class="notice notice-success is-dismissible"><p>' . __('Model IDs cleared for the selected products.', 'ardisplay') . '</p></div>';
            });
        }
    }

    /**
     * Prepare items
     */
    public function prepare_items() {
        $columns = $this->get_columns();
        $hidden = array();
        $sortable = $this->get_sortable_columns();
        
        $this->_column_headers = array($columns, $hidden, $sortable);
        
        $this->process_bulk_action();
        
        // Query parameters
        $per_page = 10;
        $current_page = $this->get_pagenum();
        
        // Get products from WooCommerce
        $args = array(
            'status' => array('publish', 'draft'),
            'limit' => $per_page,
            'page' => $current_page,
            'paginate' => true,
        );
        
        // Add search
        if (!empty($_REQUEST['s'])) {
            $args['search'] = sanitize_text_field($_REQUEST['s']);
        }
        
        // Add sorting
        if (!empty($_REQUEST['orderby'])) {
            $args['orderby'] = sanitize_text_field($_REQUEST['orderby']);
            $args['order'] = !empty($_REQUEST['order']) ? sanitize_text_field($_REQUEST['order']) : 'ASC';
        }
        
        // Get products query
        $products_query = wc_get_products($args);
        
        $this->items = $products_query->products;
        
        // Set pagination args
        $this->set_pagination_args(array(
            'total_items' => $products_query->total,
            'per_page'    => $per_page,
            'total_pages' => $products_query->max_num_pages
        ));
    }
    
    /**
     * Enqueue scripts and styles
     */
    public static function enqueue_scripts() {
        // Add error handler for blocked scripts

        // Enqueue custom CSS for the table styling
        wp_enqueue_style('ardisplay-admin-table', AR_DISPLAY_PLUGIN_URL . 'assets/css/ardisplay-table.css', array(), '1.0.0');
        wp_enqueue_script('ardisplay-admin-table', AR_DISPLAY_PLUGIN_DIR . 'admin/js/ardisplay-table.js', array('jquery'), '1.0.0', true);

        // Add localized script for AJAX
        wp_localize_script('ardisplay-admin-table', 'ardisplayTable', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('ardisplay_table_nonce')
        ));
    }

    /**
     * Render AR Model Management column content
     *
     * @param string $column_name Column identifier
     * @param int $post_id Product post ID
     */
    public function render_ardisplay_column($column_name, $post_id) {

        if ($column_name === '_ardisplay_model') {
            $model_id = get_post_meta($post_id, '_ardisplay_model_id', true);

            if ($model_id) {
                // <button class='button button-small ardisplay-change-model' data-product-id='{$post_id}' title='" . esc_attr(__('Change Model', 'ardisplay')) . "'><span class='dashicons dashicons-update'></span></button>
                // &nbsp;
                return "<div class='ardisplay-model-actions'>
                <code class='ardisplay-model-id'>Model ID: {$model_id}</code>
                <button class='button-link-delete ardisplay-clear-model' data-product-id='{$post_id}' title='" . esc_attr(__('Detach Model', 'ardisplay')) . "'>
                    <span class='dashicons dashicons-admin-links'></span>
                </button>
                </div>";
            } else {
                // return "<button class='button button-small ardisplay-change-model' data-product-id='{$post_id}' title='" . esc_attr(__('Attach Model', 'ardisplay')) . "'>" . __('Attach Model', 'ardisplay') . "</button>";
                return "No Model";
            }
        }
    }

    /**
     * Enqueue scripts for model management
     */
    public function enqueue_model_management_scripts() {
        
        wp_enqueue_style('ardisplay-table-style', plugin_dir_url(__FILE__) . 'css/ardisplay-table.css', array(), '1.0.0');
        
    }
}
