<?php

/**
 * 3D Model functionality for the ARDisplay plugin
 *
 * @link       https://ardisplay.com
 * @since      1.0.0
 *
 * @package    ARDisplay
 * @subpackage ARDisplay/admin
 */

/**
 * The admin-specific 3D model functionality of the plugin.
 *
 * Handles the admin-facing aspects of 3D model management, including
 * storing and updating models through AJAX requests.
 *
 * @package    ARDisplay
 * @subpackage ARDisplay/admin
 */
class AR_Display_3D_Model {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string    $plugin_name       Optional. The name of this plugin.
     * @param    string    $version           Optional. The version of this plugin.
     */
    public function __construct($plugin_name = 'ardisplay', $version = '1.0.0') {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        
        // Register AJAX actions
        add_action('wp_ajax_ardisplay_store_3d_model', array($this, 'store_3d_model_ajax'));
        add_action('wp_ajax_nopriv_ardisplay_store_3d_model', 'store_3d_model_ajax');
        add_action('wp_ajax_ardisplay_update_3d_model', array($this, 'update_3d_model_ajax'));
        add_action('wp_ajax_ardisplay_get_3d_model', array($this, 'get_3d_model_ajax'));
        add_action('wp_ajax_ardisplay_delete_model', array($this, 'delete_model'));
        add_action('wp_ajax_ardisplay_toggle_model_status', array($this, 'toggle_model_status'));
        add_action('wp_ajax_ardisplay_toggle_ar', array($this, 'update_ardisplay_enabled_ajax'));
        add_action('wp_ajax_ardisplay_update_product_model', array($this, 'update_product_model_ajax'));
        add_action('wp_ajax_ardisplay_clear_product_model', array($this, 'clear_product_model_ajax'));
    }

    /**
     * Handle the AJAX request to store a 3D model.
     *
     * @since    1.0.0
     */
    public function store_3d_model_ajax() {
        
        // Check nonce for security
        if (!isset($_REQUEST['nonce']) || !wp_verify_nonce($_REQUEST['nonce'], 'ardisplay_3d_model_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'ardisplay')));
            return;
        }
        
        // Get the model data from the form field
        if (!isset($_POST['model_data'])) {
            wp_send_json_error(array('message' => __('Invalid request data.', 'ardisplay')));
            return;
        }
        
        // Decode the JSON string from the form field
        $model_data = json_decode(stripslashes($_POST['model_data']), true);

        if (!$model_data) {
            wp_send_json_error(array('message' => __('Invalid request data.', 'ardisplay')));
            return;
        }
        
        // Validate required fields
        if (empty($model_data['title'])) {
            wp_send_json_error(array('message' => __('Title is required.', 'ardisplay')));
            return;
        }
        
        if (empty($model_data['variants']) || !is_array($model_data['variants']) || count($model_data['variants']) === 0) {
            wp_send_json_error(array('message' => __('At least one variant is required.', 'ardisplay')));
            return;
        }
        
        // Prepare model data for API
        $api_data = array(
            'title' => sanitize_text_field($model_data['title']),
            'description' => isset($model_data['description']) ? sanitize_textarea_field($model_data['description']) : '',
            'addToCartUrl' => isset($model_data['addToCartUrl']) ? esc_url_raw($model_data['addToCartUrl']) : '',
            'url'=> isset($model_data['url']) ? esc_url_raw($model_data['url']) : '',
            'variants' => $this->sanitize_variants($model_data['variants'])
        );
        
        // Send data to API
        $api_endpoint = '/3d-model';
        $response = AR_Display_api_key_Helper::make_api_request($api_endpoint, $api_data, 'PUT');
        
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
            return;
        }

        if (isset($model_data['product_id']) && $model_data['product_id']) {
            $product = wc_get_product($model_data['product_id']);
            if ($product) {
                if (!metadata_exists('post', $model_data['product_id'], '_ardisplay_model_id')) {
                    add_post_meta($model_data['product_id'], '_ardisplay_model_id', $response['model']['id']);
                } else {
                    update_post_meta($model_data['product_id'], '_ardisplay_model_id', $response['model']['id']);
                }
            } 
        }

        
        wp_send_json_success(array(
            'message' => __('3D model saved successfully.', 'ardisplay'),
            'model_id' => isset($response['model']['id']) ? $response['model']['id'] : null,
            'data' => $response
        ));
    }

    /**
     * Handle the AJAX request to update a 3D model.
     *
     * @since    1.0.0
     */
    public function update_3d_model_ajax() {
        // Check nonce for security
        if (!isset($_REQUEST['nonce']) || !wp_verify_nonce($_REQUEST['nonce'], 'ardisplay_3d_model_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'ardisplay')));
            return;
        }
        
        // Get model_id and model_data from form fields
        if (!isset($_POST['model_data']) || !isset($_POST['model_id'])) {
            wp_send_json_error(array('message' => __('Invalid request data.', 'ardisplay')));
            return;
        }
        
        // Decode the JSON string from the form field
        $model_data = json_decode(stripslashes($_POST['model_data']), true);
        $model_id = sanitize_text_field($_POST['model_id']);
        
        if (!$model_data) {
            wp_send_json_error(array('message' => __('Invalid request data.', 'ardisplay')));
            return;
        }
        
        // Validate required fields
        if (empty($model_data['title'])) {
            wp_send_json_error(array('message' => __('Title is required.', 'ardisplay')));
            return;
        }
        
        if (empty($model_data['variants']) || !is_array($model_data['variants']) || count($model_data['variants']) === 0) {
            wp_send_json_error(array('message' => __('At least one variant is required.', 'ardisplay')));
            return;
        }
        
        // Prepare model data for API
        $api_data = array(
            'id'=> $model_id,
            'storeProductId' => isset($model_data['product_id']) ? $model_data['product_id'] : '',
            'title' => sanitize_text_field($model_data['title']),
            'description' => isset($model_data['description']) ? sanitize_textarea_field($model_data['description']) : '',
            'addToCartUrl' => isset($model_data['addToCartUrl']) ? esc_url_raw($model_data['addToCartUrl']) : '',
            'url'=> isset($model_data['url']) ? esc_url_raw($model_data['url']) : '',
            'variants' => $this->sanitize_variants($model_data['variants'])
        );

        // Send data to API using PUT method
        $api_endpoint = '/3d-model/';
        $response = AR_Display_api_key_Helper::make_api_request($api_endpoint, $api_data, 'PATCH');
        
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
            return;
        }

        if (isset($api_data['storeProductId']) && $api_data['storeProductId']) {
            $product = wc_get_product($api_data['storeProductId']);
            if ($product) {
                if (!metadata_exists('post', $api_data['storeProductId'], '_ardisplay_model_id')) {
                    add_post_meta($api_data['storeProductId'], '_ardisplay_model_id', $model_id);
                } else {
                    update_post_meta($api_data['storeProductId'], '_ardisplay_model_id', $model_id);
                }
            } 
        }

        wp_send_json_success(array(
            'message' => __('3D model updated successfully.', 'ardisplay'),
            'model_id' => $model_id,
            'data' => $response
        ));
    }
    
    /**
     * Fetch model data from the API
     *
     * @since    1.0.0
     * @param    string    $model_id    The model ID to fetch
     * @return   array|WP_Error    Model data or error
     */
    public function get_model_data($model_id) {
        if (empty($model_id)) {
            return new WP_Error(
                'missing_model_id',
                __('Model ID is required', 'ardisplay')
            );
        }
        
        // Make direct API request using the License Helper
        $api_endpoint = '/3d-model?id=' . $model_id;
        $response = AR_Display_api_key_Helper::make_api_request($api_endpoint, array(), 'GET');
        
        if (is_wp_error($response)) {
            return $response;   
        }
        
        return $response;
    }
    
    /**
     * Handle the AJAX request to get a 3D model.
     *
     * @since    1.0.0
     */
    public function get_3d_model_ajax() {
        // Check nonce for security
        if (!isset($_REQUEST['nonce']) || !wp_verify_nonce($_REQUEST['nonce'], 'ardisplay_3d_model_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'ardisplay')));
        }
        
        // Check if model_id exists
        if (empty($_REQUEST['model_id'])) {
            wp_send_json_error(array('message' => __('Model ID is required.', 'ardisplay')));
        }
        
        $model_id = intval($_REQUEST['model_id']);
        $model_data = $this->get_model_data($model_id);
        
        if (is_wp_error($model_data)) {
            wp_send_json_error(array('message' => $model_data->get_error_message()));
        }
        
        wp_send_json_success(array(
            'message' => __('3D model fetched successfully.', 'ardisplay'),
            'model' => $model_data
        ));
    }
    
    /**
     * Handle the AJAX request to delete a 3D model.
     *
     * @since    1.0.0
     */
    public function delete_model() {
        // Check nonce for security
        if (!isset($_REQUEST['nonce']) || !wp_verify_nonce($_REQUEST['nonce'], 'ardisplay_3d_model_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'ardisplay')));
            return;
        }
        
        // Get model_id from the request
        if (!isset($_POST['model_id'])) {
            wp_send_json_error(array('message' => __('Model ID is required.', 'ardisplay')));
            return;
        }
        
        $model_id = sanitize_text_field($_POST['model_id']);
        
        // API endpoint for deleting a model
        $api_endpoint = '/3d-model';
        $response = AR_Display_api_key_Helper::make_api_request($api_endpoint, array('id' => $model_id), 'DELETE');
        
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
            return;
        }
        
        wp_send_json_success(array('message' => __('Model deleted successfully.', 'ardisplay')));
    }
    
    /**
     * Handle the AJAX request to toggle a 3D model's status (active/inactive).
     *
     * @since    1.0.0
     */
    public function toggle_model_status() {
        // Check nonce for security
        if (!isset($_REQUEST['nonce']) || !wp_verify_nonce($_REQUEST['nonce'], 'ardisplay_3d_model_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'ardisplay')));
            return;
        }

        
        // Get model_id and status from the request
        if (!isset($_POST['model_id']) || !isset($_POST['enabled'])) {
            wp_send_json_error(array('message' => __('Model ID and status are required.', 'ardisplay')));
            return;
        }
        
        $model_id = sanitize_text_field($_POST['model_id']);
        $status = (bool) $_POST['enabled'];
        
        // Prepare data for API
        $api_data = array(
            'modelId'=> $model_id,
            'enabled' => $status
        );
        
        // API endpoint for updating a model's status
        $api_endpoint = '3d-model/store/status';
        $response = AR_Display_api_key_Helper::make_api_request($api_endpoint, $api_data, 'POST');
        
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => $response->get_error_message()));
            return;
        }
        
        wp_send_json_success(array('message' => __('Model status updated successfully.', 'ardisplay')));
    }

    /**
     * Update the _ardisplay_enabled post meta based on the given value
     *
     * @since    1.0.0
     * @param    int    $product_id    The product ID to update
     * @param    boolean    $enabled    The new value for _ardisplay_enabled
     * @return   void
     */
    public function update_ardisplay_enabled_ajax() {
        if (!isset($_REQUEST['nonce']) || !wp_verify_nonce($_REQUEST['nonce'], 'ardisplay-toggle-ar')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'ardisplay')));
            return;
        }
        
        if (!isset($_POST['product_id']) || !isset($_POST['enabled'])) {
            wp_send_json_error(array('message' => __('Product ID and status are required.', 'ardisplay')));
            return;
        }
        
        $product_id = (int) $_POST['product_id'];
        $enabled = (bool) $_POST['enabled'];
        
        $this->update_ardisplay_enabled($product_id, $enabled);
        
        wp_send_json_success(array('message' => __('Model status updated successfully.', 'ardisplay')));
    }

    /**
     * Generate a function to update the _ardisplay_enabled post meta based on the given value
     *
     * @since    1.0.0
     * @param    int    $product_id    The product ID to update
     * @param    boolean    $enabled    The new value for _ardisplay_enabled
     * @return   void
     */
    private function update_ardisplay_enabled($product_id, $enabled) {
        update_post_meta($product_id, '_ardisplay_enabled', $enabled);
    }

    /**
     * AJAX handler to update a product's model ID
     *
     * @since    1.0.0
     */
    public function update_product_model_ajax() {
        // Check nonce for security
        check_ajax_referer('ardisplay-update-product-model', 'nonce');

        // Verify user capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('You do not have permission to update models.', 'ardisplay')));
        }

        // Sanitize and validate input
        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        $model_id = isset($_POST['model_id']) ? sanitize_text_field($_POST['model_id']) : '';

        if (!$product_id || !$model_id) {
            wp_send_json_error(array('message' => __('Invalid product or model ID.', 'ardisplay')));
        }

        // Update product meta
        update_post_meta($product_id, '_ardisplay_model_id', $model_id);

        wp_send_json_success(array(
            'message' => __('Product model updated successfully.', 'ardisplay'),
            'model_id' => $model_id
        ));
    }

    /**
     * AJAX handler to clear a product's model ID
     *
     * @since    1.0.0
     */
    public function clear_product_model_ajax() {
        // Check nonce for security
        check_ajax_referer('ardisplay-clear-product-model', 'nonce');

        // Verify user capabilities
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __('You do not have permission to update models.', 'ardisplay')));
        }

        // Sanitize and validate input
        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;

        if (!$product_id) {
            wp_send_json_error(array('message' => __('Invalid product ID.', 'ardisplay')));
        }

        // Clear product meta
        delete_post_meta($product_id, '_ardisplay_model_id');
        delete_post_meta($product_id, '_ardisplay_enabled');

        wp_send_json_success(array(
            'message' => __('Product model cleared successfully.', 'ardisplay')
        ));
    }

    /**
     * Sanitize variants data.
     *
     * @since    1.0.0
     * @param    array    $variants    The array of variants to sanitize.
     * @return   array    The sanitized variants.
     */
    private function sanitize_variants($variants) {
        $sanitized = array();

        
        foreach ($variants as $variant) {
            if (isset($variant['name'])) {
                $sanitized_variant = array(
                    'model'=> isset($variant['model']) ? sanitize_text_field($variant['model']) : '',
                    'name' => sanitize_text_field($variant['name']),
                    'url' => isset($variant['url']) ? esc_url_raw($variant['url']) : '',
                    'posterFileUrl' => isset($variant['posterFileUrl']) ? esc_url_raw($variant['posterFileUrl']) : '',
                    'placement' => isset($variant['placement']) ? sanitize_text_field($variant['placement']) : 'floor',
                    'iosUrl' => isset($variant['iosUrl']) ? esc_url_raw($variant['iosUrl']) : '',
                    'iosId' => isset($variant['iosId']) ? sanitize_text_field($variant['iosId']) : '',
                    'androidId' => isset($variant['androidId']) ? sanitize_text_field($variant['androidId']) : '',
                    'posterFileId' => isset($variant['posterFileId']) ? sanitize_text_field($variant['posterFileId']) : '',
                );
                
                // Handle sizes
                if (isset($variant['sizes']) && is_array($variant['sizes'])) {
                    $sanitized_variant['sizes'] = array();
                    foreach ($variant['sizes'] as $size) {
                        if (isset($size['label'])) {
                            $sanitized_variant['sizes'][] = array(
                                'label' => sanitize_text_field($size['label']),
                                'width' => isset($size['width']) ? strval($size['width']) : '0',
                                'height' => isset($size['height']) ? strval($size['height']) : '0',
                                'depth' => isset($size['depth']) ? strval($size['depth']) : '0'
                            );
                        }
                    }
                }
                
                $sanitized[] = $sanitized_variant;
            }
        }
        
        return $sanitized;
    }
}
